<?php

defined('BASEPATH') OR exit('No direct script access allowed');

/* * *****************Info.php**********************************
 * @product name    : Global Multi School Management System Express
 * @type            : Class
 * @class name      : Info
 * @description     : Manage school academic info.  
 * @author          : Codetroopers Team 	
 * @url             : https://themeforest.net/user/codetroopers      
 * @support         : yousuf361@gmail.com	
 * @copyright       : Codetroopers Team	 	
 * *********************** *********************************** */

class Info extends MY_Controller {

    public $data = array();
    
    
    function __construct() {
        parent::__construct();
         $this->load->model('Info_Model', 'info', true);               
    }

    
    
    /*****************Function index**********************************
    * @type            : Function
    * @function name   : index
    * @description     : Load "Info List" user interface                 
    *                        
    * @param           : null
    * @return          : null 
    * ********************************************************** */
    public function index($school_id = null) {
        
        check_permission(VIEW);

        $this->data['info_list'] = $this->info->get_info_list($school_id); 
        $this->data['filter_school_id'] = $school_id;
        $this->data['schools'] = $this->schools;
        
        $this->data['list'] = TRUE;
        $this->layout->title( $this->lang->line('manage_info'). ' | ' . SMS);
        $this->layout->view('info/index', $this->data);            
       
    }

    
    /*****************Function add**********************************
    * @type            : Function
    * @function name   : add
    * @description     : Load "Add new Info" user interface                 
    *                    and store "Info" into database 
    * @param           : null
    * @return          : null 
    * ********************************************************** */
    public function add() {

        check_permission(ADD);
         
        if ($_POST) {
            $this->_prepare_info_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_info_data();

                $insert_id = $this->info->insert('info', $data);
                if ($insert_id) {
                    
                    create_log('Has been created a info : '.$data['title']);
                    success($this->lang->line('insert_success'));
                    redirect('announcement/info/index/'.$data['school_id']);
                    
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('announcement/info/add');
                }
            } else {
                error($this->lang->line('insert_failed'));
                $this->data['post'] = $_POST;
            }
        }
        
        $this->data['info_list'] = $this->info->get_info_list(); 
        $this->data['schools'] = $this->schools;
         
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add'). ' | ' . SMS);
        $this->layout->view('info/index', $this->data);
    }

    
    /*****************Function edit**********************************
    * @type            : Function
    * @function name   : edit
    * @description     : Load Update "Info" user interface                 
    *                    with populated "Info" value 
    *                    and update "Info" database    
    * @param           : $id integer value
    * @return          : null 
    * ********************************************************** */
    public function edit($id = null) {       
       
         check_permission(EDIT);
        
         if(!is_numeric($id)){
            error($this->lang->line('unexpected_error'));
            redirect('announcement/info/index');    
        }
        
        if ($_POST) {
            $this->_prepare_info_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_info_data();
                $updated = $this->info->update('info', $data, array('id' => $this->input->post('id')));

                if ($updated) {
                    
                     create_log('Has been updated a info : '.$data['title']);
                    
                    success($this->lang->line('update_success'));
                    redirect('announcement/info/index/'.$data['school_id']);                   
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('announcement/info/edit/' . $this->input->post('id'));
                }
            } else {
                error($this->lang->line('update_failed'));
                $this->data['info'] = $this->info->get_single('info', array('id' => $this->input->post('id')));
            }
        }
        
        if ($id) {
            $this->data['info'] = $this->info->get_single('info', array('id' => $id));

            if (!$this->data['info']) {
                 redirect('announcement/info/index');
            }
        }

        $this->data['info_list'] = $this->info->get_info_list($this->data['info']->school_id);  
        $this->data['school_id'] = $this->data['info']->school_id;
        $this->data['filter_school_id'] = $this->data['info']->school_id;
        $this->data['schools'] = $this->schools;
        
        $this->data['edit'] = TRUE;       
        $this->layout->title($this->lang->line('edit') . ' | ' . SMS);
        $this->layout->view('info/index', $this->data);
    }
    
    
    /*****************Function view**********************************
    * @type            : Function
    * @function name   : view
    * @description     : Load user interface with specific info data                 
    *                       
    * @param           : $id integer value
    * @return          : null 
    * ********************************************************** */
    public function view($id = null){
        
         check_permission(VIEW);
         if(!is_numeric($id)){
            error($this->lang->line('unexpected_error'));
            redirect('announcement/info/index');    
        }
        
        $this->data['info_list'] = $this->info->get_info_list(); 
        
        $this->data['info'] = $this->info->get_single('info', array('id' => $id));
        $this->data['detail'] = TRUE;       
        $this->layout->title($this->lang->line('view'). ' ' . $this->lang->line('info'). ' | ' . SMS);
        $this->layout->view('info/index', $this->data);
    }

                   
     /*****************Function get_single_info**********************************
     * @type            : Function
     * @function name   : get_single_info
     * @description     : "Load single info information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */
    public function get_single_info(){
        
       $info_id = $this->input->post('info_id');
       
       $this->data['info'] = $this->info->get_single_info($info_id);
       echo $this->load->view('info/get-single-info', $this->data);
    }
    
    /*****************Function _prepare_info_validation**********************************
    * @type            : Function
    * @function name   : _prepare_info_validation
    * @description     : Process "Info" user input data validation                 
    *                       
    * @param           : null
    * @return          : null 
    * ********************************************************** */
    private function _prepare_info_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        
        $this->form_validation->set_rules('school_id', $this->lang->line('school_name'), 'trim|required');   
        $this->form_validation->set_rules('title', $this->lang->line('title'), 'trim|required|callback_title');   
        $this->form_validation->set_rules('date', $this->lang->line('date'), 'trim|required');   
        $this->form_validation->set_rules('info', $this->lang->line('info'), 'trim|required');   
        $this->form_validation->set_rules('image', $this->lang->line('image'), 'trim|callback_image');   
    }
   
    
    /*****************Function title**********************************
    * @type            : Function
    * @function name   : title
    * @description     : Unique check for "info title" data/value                  
    *                       
    * @param           : null
    * @return          : boolean true/false 
    * ********************************************************** */ 
   public function title()
   {             
      if($this->input->post('id') == '')
      {   
          $info = $this->info->duplicate_check($this->input->post('school_id'),$this->input->post('title'), $this->input->post('date')); 
          if($info){
                $this->form_validation->set_message('title', $this->lang->line('already_exist'));         
                return FALSE;
          } else {
              return TRUE;
          }          
      }else if($this->input->post('id') != ''){   
         $info = $this->info->duplicate_check($this->input->post('school_id'), $this->input->post('title'),$this->input->post('date'), $this->input->post('id')); 
          if($info){
                $this->form_validation->set_message('title', $this->lang->line('already_exist'));         
                return FALSE;
          } else {
              return TRUE;
          }
      }   
   }

   
   
    /*****************Function image**********************************
    * @type            : Function
    * @function name   : image
    * @description     : Check image format validation                  
    *                       
    * @param           : null
    * @return          : boolean true/false 
    * ********************************************************** */ 
   public function image()
   {   
        if($_FILES['image']['name']){
            
            list($width, $height) = getimagesize($_FILES['image']['tmp_name']);
            if((!empty($width)) && $width > 750 || $height > 500){
                $this->form_validation->set_message('image', $this->lang->line('please_check_image_dimension'));
                return FALSE;
            }
            
            $name = $_FILES['image']['name'];
            $ext = pathinfo($name, PATHINFO_EXTENSION);
            if($ext == 'jpg' || $ext == 'jpeg' || $ext == 'png' || $ext == 'gif'){
                return TRUE;
            } else {
                $this->form_validation->set_message('image', $this->lang->line('select_valid_file_format'));         
                return FALSE; 
            }
        }       
   }
   
   
   /*****************Function resume**********************************
    * @type            : Function
    * @function name   : resume
    * @description     : validate resume                  
    *                       
    * @param           : null
    * @return          : boolean true/false 
    * ********************************************************** */
    public function resume() {
        if ($_FILES['resume']['name']) {
            $name = $_FILES['resume']['name'];
            $ext = pathinfo($name, PATHINFO_EXTENSION);
            if ($ext == 'pdf' || $ext == 'doc' || $ext == 'docx' || $ext == 'ppt' || $ext == 'pptx' || $ext == 'txt') {
                return TRUE;
            } else {
                $this->form_validation->set_message('resume', $this->lang->line('select_valid_file_format'));
                return FALSE;
            }
        }
    }
    
   
   
   
    /*****************Function _get_posted_info_data**********************************
    * @type            : Function
    * @function name   : _get_posted_info_data
    * @description     : Prepare "Info" user input data to save into database                  
    *                       
    * @param           : null
    * @return          : $data array(); value 
    * ********************************************************** */
    private function _get_posted_info_data() {

        $items = array();
        $items[] = 'school_id';
        $items[] = 'title';
        $items[] = 'info';
        $items[] = 'is_view_on_web';
        $data = elements($items, $_POST);  
      
        $data['date'] = date('Y-m-d', strtotime($this->input->post('date')));
        
        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['status'] = 1;
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();                       
        }
        
        if(isset($_FILES['image']['name'])){  
            $data['image'] =  $this->_upload_image();
          }

	
        if($_FILES['resume']['name']) {
            $data['resume'] = $this->_upload_resume();
        }  
      

        return $data;
    }

    
    
    /*****************Function _upload_image**********************************
    * @type            : Function
    * @function name   : _upload_image
    * @description     : Process to upload info image and return image name                 
    *                       
    * @param           : null
    * @return          : $return_image  string value 
    * ********************************************************** */
      private function _upload_image(){
       
        $prev_image     = $this->input->post('prev_image');
        $image          = $_FILES['image']['name'];
        $image_type     = $_FILES['image']['type'];
        $return_image   = '';
        if ($image != "") {
            if ($image_type == 'image/jpeg' || $image_type == 'image/pjpeg' ||
                    $image_type == 'image/jpg' || $image_type == 'image/png' ||
                    $image_type == 'image/x-png' || $image_type == 'image/gif') {

                $destination = 'assets/uploads/info/';               

                $file_type  = explode(".", $image);
                $extension  = strtolower($file_type[count($file_type) - 1]);
                $image_path = 'info-'.time() . '-sms.' . $extension;

                move_uploaded_file($_FILES['image']['tmp_name'], $destination . $image_path);

                // need to unlink previous image
                if ($prev_image != "") {
                    if (file_exists($destination . $prev_image)) {
                        @unlink($destination . $prev_image);
                    }
                }

                $return_image = $image_path;
            }
        } else {
            $return_image = $prev_image;
        }

        return $return_image;
    }
    
	
	
	/*****************Function _upload_resume**********************************
    * @type            : Function
    * @function name   : _upload_resume
    * @description     : process to upload info profile resume in the server                  
    *                     and return resume file name  
    * @param           : null
    * @return          : $return_resume string value 
    * ********************************************************** */
    private function _upload_resume() {
        $prev_resume = $this->input->post('prev_resume');
        $resume = $_FILES['resume']['name'];
        $resume_type = $_FILES['resume']['type'];
        $return_resume = '';

        if ($resume != "") {
            if ($resume_type == 'application/vnd.openxmlformats-officedocument.wordprocessingml.document' ||
                    $resume_type == 'application/msword' || $resume_type == 'text/plain' ||
                    $resume_type == 'application/vnd.ms-office' || $resume_type == 'application/pdf') {

                $destination = 'assets/uploads/info-resume/';

                $file_type = explode(".", $resume);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $resume_path = 'resume-' . time() . '-sms.' . $extension;

                move_uploaded_file($_FILES['resume']['tmp_name'], $destination . $resume_path);

                // need to unlink previous photo
                if ($prev_resume != "") {
                    if (file_exists($destination . $prev_resume)) {
                        @unlink($destination . $prev_resume);
                    }
                }

                $return_resume = $resume_path;
            }
        } else {
            $return_resume = $prev_resume;
        }

        return $return_resume;
    }

	
	
	
    
     /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Info" from database                  
     *                    and unlink info image from server   
     * @param           : $id integer value
     * @return          : null 
     * ********************************************************** */
    public function delete($id = null) {
        
         check_permission(DELETE);
         
        if(!is_numeric($id)){
            error($this->lang->line('unexpected_error'));
            redirect('announcement/info/index');    
        }  
        
        $info = $this->info->get_single('info', array('id' => $id));
         
        if ($this->info->delete('info', array('id' => $id))) {  
            
            // delete info resume and image
            $destination = 'assets/uploads/';
            if (file_exists( $destination.'/info/'.$info->image)) {
                @unlink($destination.'/info/'.$info->image);
            }
            
			  // delete teacher resume and photo
            $destination = 'assets/uploads/';
            if (file_exists($destination . '/info-resume/' . $info->resume)) {
                @unlink($destination . '/info-resume/' . $info->resume);
            }
			
			
			
            create_log('Has been deleted a info : '.$info->title); 
            
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect('announcement/info/index/'.$info->school_id); 
    }

}
